/*
 * colorout.c: make stderr and stdout colorfull on 
 * terminals that understand ANSI-color-sequences
 *
 * Copyright (C) 2001 by M G Berberich 
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h> 
#include <poll.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
#include <string.h>

#ifndef FALSE
#define      FALSE   (0)
#endif
#ifndef TRUE
#define      TRUE   (1)
#endif

#define NAME "colourout"

#define OUT_COLOR "\033[0m"
#define ERR_COLOR "\033[31m"
#define DEFAULT_COLOR "\033[0m"

int main(int argc, char **argv)
{
  int cid, terminated=FALSE;
  int outpipe[2], errpipe[2];
  const int bufferlen = 20;
  char out[bufferlen+1], err[bufferlen+1];
  struct pollfd fds[2];

  pipe(outpipe);
  pipe(errpipe);

  cid = fork();
  if(cid==0) { /* child */

    // stdout
    dup2(outpipe[1], 1);
    close(outpipe[0]);
    close(outpipe[1]);

    // stderr
    dup2(errpipe[1], 2);
    close(errpipe[0]);
    close(errpipe[1]); 

    execvp(argv[1], &argv[1]);
    
    fprintf(stderr, NAME ": %s\n", strerror(errno));
    exit(1);

  } else if (cid==-1){ /* error */
    fprintf(stderr, NAME ": %s\n", strerror(errno));
  }
  
  /* parent */

  close(outpipe[1]);
  close(errpipe[1]);

  fds[0].fd = outpipe[0];
  fds[0].events = POLLIN|POLLPRI|POLLHUP;
  fds[1].fd = errpipe[0];
  fds[1].events = POLLIN|POLLPRI|POLLHUP;
  
  while(1) {
    int pres;
    int status;

    if (!terminated && waitpid(cid, &status, WNOHANG) == cid) 
      terminated = TRUE;

    pres = poll(fds, 2, 100);
    printf("--- %d: %d %d\n", pres, fds[0].revents , fds[1].revents);
    if (pres>0) {
      if ((fds[0].revents | fds[1].revents) & POLLERR) {
	fprintf(stderr, NAME ": %s\n", strerror(errno));
	kill(cid, SIGINT);
	wait(0);
	exit(1);
      }
      if (fds[0].revents & (POLLIN|POLLPRI)) {
	int n = read(outpipe[0], out, bufferlen);
	if (n>0)
	  printf("%s%.*s%s", OUT_COLOR, n, out, DEFAULT_COLOR);
      }
      if (fds[1].revents & (POLLIN|POLLPRI)) {
	int n = read(errpipe[0], err, bufferlen);
	if (n>0)
	  printf("%s%.*s%s", ERR_COLOR, n, err, DEFAULT_COLOR);
      } 
      if (fds[0].revents == POLLHUP) {
	fds[0].fd = -1;
	close(fds[0].fd);
      }
      if (fds[1].revents == POLLHUP) {
	fds[1].fd = -1;
	close(fds[1].fd);
      } 
    } else if (pres<0) {
      fprintf(stderr, NAME ": %s\n", strerror(errno));
    } else {
      if (terminated) return WEXITSTATUS(status);
    }
  }
}
