/* GNOME modemlights applet
 * (C) 2000 John Ellis
 *
 * Authors: John Ellis
 *          Martin Baulig
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place - Suite 330, Cambridge, MA 
 * 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "modemlights.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>

#ifdef __OpenBSD__
#include <net/if_ppp.h>
#endif

#ifdef __linux__
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/if_frad.h>
#include <linux/ppp_defs.h>
#include <linux/ppp.h>

#include <linux/isdn.h>

#ifndef ISDN_MAX_CHANNELS
#define ISDN_MAX_CHANNELS 64
#endif

static unsigned long *isdn_stats = NULL;
#endif

#ifndef FALSE
#define FALSE 0
#define TRUE 1
#endif

static const int verify_lock_file = TRUE;

int is_Modem_on(const char *lock_file)
{
	FILE *f = 0;
	char buf[64];
	pid_t pid = -1;

	f = fopen(lock_file, "r");

	if(!f) return FALSE;

	if (verify_lock_file)
		{
		if (fgets(buf, sizeof(buf), f) == NULL)
			{
			fclose(f);
			return FALSE;
			}
		}

	fclose(f);

	if (verify_lock_file)
		{
		pid = (pid_t)strtol(buf, NULL, 10);
		if (pid < 1 || (kill (pid, 0) == -1 && errno != EPERM)) return FALSE;
		}

	return TRUE;
}

int is_ISDN_on()
{
#ifdef __linux__

	/* Perhaps I should try to explain this code a little bit.
	 *
	 * ------------------------------------------------------------
	 * This is from the manpage of isdninfo(4):
	 *
	 * DESCRIPTION
	 *   /dev/isdninfo  is  a character device with major number 45
	 *   and minor number 255.  It delivers status information from
	 *   the Linux ISDN subsystem to user level.
	 *
	 * DATA FORMAT
	 *   When  reading  from this device, the current status of the
	 *   Linux ISDN subsystem is delivered in 6 lines of text. Each
	 *   line  starts  with  a  tag  string followed by a colon and
	 *   whitespace. After that the status values are appended sep-
	 *   arated by whitespace.
	 *
	 *   flags  is the tag of line 5. In this line for every driver
	 *          slot, it's B-Channel status is shown. If no  driver
	 *          is  registered  in a slot, a ? is shown.  For every
	 *          established B-Channel of the driver, a bit  is  set
	 *          in  the  shown value. The driver's first channel is
	 *          mapped to bit 0, the second channel to bit 1 and so
	 *          on.
	 * ------------------------------------------------------------
	 *
	 * So we open /dev/isdninfo, discard the first four lines of text
	 * and then check whether we have something that is not `0' or `?'
	 * in one of the flags fields.
	 *
	 * Sounds complicated, but I don't see any other way to check whether
	 * we are connected. Also, this is the method some other ISDN tools
	 * for Linux use.
	 *
	 * Martin
	 *
	 * ---
	 *
	 * With Linux kernel 2.4, glibc 2.2 (*) and certain providers with
	 * long phone numbers, the contents of /dev/isdninfo can be more than
	 * 1024 bytes and not be properly processed by multiple subsequent
	 * fgets()'s, so we (try to atomically) read() BUFSIZE bytes into
	 * buffer[] and process this instead of /dev/isdninfo directly.
	 *
	 * (*): dont't nail me on this combination
	 *
	 * Nils
	 */

	char buffer [BUFSIZ], *p;
	int i;
	int fd;
	int length;

	fd = open ("/dev/isdninfo", O_RDONLY | O_NDELAY);

	if (fd < 0) {
		perror ("/dev/isdninfo");
		return FALSE;
	}

	if ((length = read (fd, buffer, BUFSIZ - 1)) < 0) {
		perror ("/dev/isdninfo");
		close (fd);
		return FALSE;
	}

	buffer[length+1] = (char)0;

	p = buffer;

	/* Seek for the fifth line */
	for (i = 1; i < 5; i++) {
		if ((p = strchr (p, '\n')) == NULL) {
			close (fd);
			return FALSE;
		}
		p++;
	}

	/* *p is the first character of the fifth line now */
	if (strncmp (p, "flags:", 6)) {
		close (fd);
		return FALSE;
	}

	p += 6;

	while (*p && (*p != '\n')) {
		char *end = p;

		if (isspace (*p)) {
			p++;
			continue;
		}

		for (end = p; *end && !isspace (*end); end++)
			;

		if (*end == 0)
			break;
		else
			*end = 0;

		if (!strcmp (p, "?") || !strcmp (p, "0")) {
			p = end+1;
			continue;
		}

		close (fd);
		return TRUE;
	}

	close (fd);

	return FALSE;
#else
	return FALSE;
#endif
}

int get_modem_stats(int *in, int *out, int ip_socket, const char *device_name)
{
	struct 	ifreq ifreq;
	struct 	ppp_stats stats;

	memset(&ifreq, 0, sizeof(ifreq));
	strncpy(ifreq.ifr_name, device_name, IFNAMSIZ);
	ifreq.ifr_name[IFNAMSIZ-1] = '\0';
	ifreq.ifr_ifru.ifru_data = (caddr_t)&stats;

#ifdef SIOCGPPPSTATS
	if (ioctl(ip_socket, SIOCGPPPSTATS, (caddr_t)&ifreq) < 0)
#else
	if (TRUE)
#endif
		{
		/* failure means ppp is not up */
		*in = *out = 0;
		return FALSE;
		}
	else
		{
		*in = stats.p.ppp_ibytes;
		*out = stats.p.ppp_obytes;
		return TRUE;
		}
}

int get_ISDN_stats(int *in, int *out)
{
#ifdef __linux__
	int fd, i;
	unsigned long *ptr;

	*in = *out = 0;

	if (!isdn_stats)
		isdn_stats = calloc (ISDN_MAX_CHANNELS * 2, sizeof(unsigned long));

	fd = open("/dev/isdninfo", O_RDONLY);

	if (fd < 0)
		return FALSE;

	if ((ioctl (fd, IIOCGETCPS, isdn_stats) < 0) && (errno != 0)) {
		close (fd);
		
		return FALSE;
	}

	for (i = 0, ptr = isdn_stats; i < ISDN_MAX_CHANNELS; i++) {
		*in  += *ptr++; *out += *ptr++;
	}

	close (fd);

	return TRUE;

#else
	*in = *out = 0;

	return FALSE;
#endif
}

int get_ISDN_connect_time(int recalc_start)
{
	/* this is a bad hack just to get some (not very accurate) timing */
	static time_t start_time = (time_t)0;

	if (recalc_start)
		{
		start_time = time(0);
		}

	if (start_time != (time_t)0)
		return (int)(time(0) - start_time);
	else
		return -1;
}

int get_modem_connect_time(int recalc_start, const char *lock_file)
{
	static time_t start_time = (time_t)0;
	struct stat st;

	if (recalc_start)
		{
		if (stat (lock_file, &st) == 0)
			start_time = st.st_mtime;
		else
			start_time = (time_t)0;
		}

	if (start_time != (time_t)0)
		return (int)(time(0) - start_time);
	else
		return -1;
}
