/*
 * (c) 2004 M G Berberich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file called COPYING; if not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <time.h>

#include <qlayout.h>
#include <qvbox.h>
#include <qiconset.h> 
#include <qsizepolicy.h> 

#include <kapplication.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <krun.h>
#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <kaboutapplication.h>
#include <kstandarddirs.h>
#include <kpopupmenu.h>
#include <kdebug.h>

#include "statusbutton.h"
#include "light.h"
#include "graph.h"
#include "modemlights.h"
#include "kmodemlightsdlg.h"
#include "kmodemlights.h"
#include "kmodemlights.moc"

extern "C" {
  KPanelApplet *init(QWidget *parent, const QString& configFile)
  {
    KGlobal::locale()->insertCatalogue("kmodemlights");
    return new KModemLightsApplet(configFile, KPanelApplet::Normal,
			 KPanelApplet::About|KPanelApplet::Preferences,
			 parent, "kmodemlightapplet");
  }
}

KModemLightsApplet::KModemLightsApplet(const QString& configFile, Type type, 
				     int actions, QWidget *parent, 
				     const char *name)
  : KPanelApplet(configFile, type, actions, parent, name),
    aboutData(0),
    timerID(0),
    ffhAppletHeight(-1),
    useISDN(false),
    loadCounter(0),
    lastTimeWasConnected(false),
    modemWasOn(false),
    startTime(time(0)),
    oldRxBytes(-1), RxBytesShowed(-1)
{
  KGlobal::iconLoader()->addAppDir("kmodemlights");

  socket = ::socket(AF_INET, SOCK_DGRAM, 0);

  // The applett itself

  // Button
  button = new StatusButton(this);
  button->setStatus(StatusButton::Off);
  button->setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Minimum));
  QObject::connect(button, SIGNAL(clicked()), this, SLOT(dial()));

  // graph and two lights
  lightRx = new Light(QColor(0x00FF00), QColor(0x004400), this);
  lightTx = new Light(QColor(0xFF0000), QColor(0x440000), this);
  graph = new Graph(this);
  graph->setMinimumWidth(22);

  QHBoxLayout *graphPart = new QHBoxLayout();
  QVBoxLayout *vbox1 = new QVBoxLayout();
  graphPart->addWidget(graph);
  graphPart->addLayout(vbox1);
  vbox1->addWidget(lightRx);
  vbox1->addWidget(lightTx);

  // textual Displays
  showRx = new QLabel("", this);
  showRx->setAlignment(AlignRight|AlignVCenter);
  showTime = new QLabel("*", this);
  showTime->setAlignment(AlignRight|AlignVCenter);
  QVBoxLayout *textualPart = new QVBoxLayout();
  textualPart->addWidget(showRx);
  textualPart->addWidget(showTime);

  // layouts
  infoPart = new QBoxLayout(QBoxLayout::TopToBottom);
  infoPart->addLayout(graphPart);
  infoPart->addLayout(textualPart);
  page = new QHBoxLayout(this);
  page->setSpacing(1);
  page->addWidget(button);
  page->addLayout(infoPart);

  // popupMenu
  popup = new KPopupMenu();
  popup->insertTitle(SmallIcon("kmodemlighs"), i18n("kmodemlights"));
  popup->insertItem(i18n("&About kmodemlights..."), this,
		    SLOT(about()));
  popup->insertSeparator();
  popup->insertItem(SmallIcon("configure"), i18n("&Settings..."), this,
		    SLOT(preferences()));

  readConfig();
  
  timerID = startTimer(1000/updateFrequency);
}

KModemLightsApplet::~KModemLightsApplet()
{
  killTimer(timerID);
#if 1
  if(socket != -1) ::close(socket);
#endif
  if(aboutData) delete aboutData;
}

int KModemLightsApplet::widthForHeight(int height) const
{
  int fontheight, widgetheight;

  // calc Fontheight
  if (height>=42) {	// two-row-layout
    widgetheight = height/4;
    fontheight = widgetheight;
    if(infoPart->direction() != QBoxLayout::TopToBottom)
      infoPart->setDirection(QBoxLayout::TopToBottom);
  } else {		// single-row-layout
    widgetheight = height/2;
    fontheight = widgetheight;
    if(infoPart->direction() != QBoxLayout::LeftToRight)
      infoPart->setDirection(QBoxLayout::LeftToRight);    
  }

  // take the configured KDE fixed font
  QFont font(KGlobalSettings::fixedFont());
  font.setPixelSize(fontheight);
  // and set it on the text-widgets
  showRx->setFont(font);
  showRx->setMaximumHeight(widgetheight);
  showTime->setFont(font);
  showTime->setMaximumHeight(widgetheight);
  
  if(widgetheight>12) {
    lightRx->setBig(true);
    lightTx->setBig(true);
  } else {
    lightRx->setBig(false);
    lightTx->setBig(false);
  }

  // guess the maximum size by setting a label to 
  // 4 digits 2 characters(b,m,' or ")
  showTime->setText("00'00\"");

  return this->sizeHint().width();;
}

int KModemLightsApplet::heightForWidth(int) const
{
  // fixme: this does not work yet
  // there is no concept for vertical layout at all

  QFont font(KGlobalSettings::fixedFont());
  font.setPixelSize(10);

  showRx->setFont(font);
  showRx->setMaximumHeight(10);
  showTime->setFont(font);
  showTime->setMaximumHeight(10);

  return 42;
}

void KModemLightsApplet::mousePressEvent(QMouseEvent *e)
{
  if (e->button() == RightButton) {
    popup->popup(mapToGlobal(e->pos()));
    popup->exec();
  }
}

static inline QString formatBytes(int n)
{
  if(n<10000) 
    return QString().sprintf("%4db", n);
  else if(n<10000000) 
    return QString().sprintf("%4dK", n/1024);
  else
    return QString().sprintf("%4dM", n/(1024*1024));
}

void KModemLightsApplet::timerEvent(QTimerEvent *)
{
  int rx, tx;
  bool lightRx = false;
  bool lightTx = false;

  loadCounter++;

  if (isConnected()) {
    // Timer
    if (!lastTimeWasConnected) {
      getConnectTime(true); // reset start time
      lastTimeWasConnected = TRUE;
    }
    updateTimer();

    // button and lights
    if (!getStats(rx, tx) || (rx == 0 && tx == 0)) {
      oldRx = oldTx = 0;
      button->setStatus(StatusButton::Wait);
    } else {
      button->setStatus(StatusButton::On);
      if (rx > oldRx) lightRx = true;
      if (tx > oldTx) lightTx = true;
    }
    updateLights(lightRx, lightTx);
    oldRx = rx;
    oldTx = tx;

    // (all 1 sec)
    if (loadCounter % updateFrequency == 0) {
      updateRx(rx);

      // (all 2 sec)
      if (loadCounter % (updateFrequency * 2) == 0) {
	
	// (all 20 sec)
	if (loadCounter % (updateFrequency * 20) == 0) {
	  //	update_tooltip(TRUE, rx, tx);
	}
	
	if (!modemWasOn) {
	  loadRx = rx; // reset loadRx/loadTx
	  loadTx = tx;
	  //	update_tooltip(TRUE, rx, tx);
	  modemWasOn = TRUE;
	}
	
	updateLoad(rx-loadRx, tx-loadTx);
	loadRx = rx;
	loadTx = tx;
      }
    }
  } else {
    button->setStatus(StatusButton::Off);
    updateRx(-1);
    updateLights(false, false);
    
    // (all 2 sec)
    if (loadCounter % (updateFrequency * 2) == 0) {
      updateLoad(0, 0);
    }
    if (modemWasOn) {
      //      update_tooltip(FALSE,0,0);
      modemWasOn = FALSE;
    }
    if (lastTimeWasConnected) lastTimeWasConnected = FALSE;
  }
}


bool KModemLightsApplet::isConnected()
{
  if (useISDN)
    return is_ISDN_on();
  else
    return is_Modem_on(lockFile.ascii());
}

bool KModemLightsApplet::getStats(int &in, int &out)
{
  if (useISDN)
    return get_ISDN_stats(&in, &out);
  else
    return get_modem_stats(&in, &out, socket, deviceName.ascii());
}

int KModemLightsApplet::getConnectTime(bool recalcStart)
{
  if (useISDN)
    return get_ISDN_connect_time(recalcStart);
  else
    return get_modem_connect_time(recalcStart, lockFile.ascii());
}

void KModemLightsApplet::updateLights(bool rx, bool tx)
{
  lightRx->setStatus(rx);
  lightTx->setStatus(tx);
}

void KModemLightsApplet::updateLoad(unsigned int rx, unsigned int tx)
{
  graph->addValues(rx, tx);
}

void KModemLightsApplet::updateRx(int rxBytes)
{
  // reset
  if (rxBytes == -1) {
    oldRxBytes = -1;
    showRx->setText("");
    return;
  }
  // init
  if (oldRxBytes == -1) {
    oldRxBytes = rxBytes;
    return;
  }

  if (oldRxBytes > rxBytes) {
    oldRxBytes = rxBytes;
    return;
  }

  int rx = rxBytes - oldRxBytes;
  oldRxBytes = rxBytes;

  if (rx == RxBytesShowed) return;

  showRx->setText(formatBytes(rx));
  RxBytesShowed = rx;

}

void  KModemLightsApplet::updateTimer()
{
  int newTimer = getConnectTime(false);
  if (newTimer != oldTimer) {
    if (newTimer>3600) { /* HH:MM */
      newTimer /= 60; 
      int a = newTimer / 60;
      int b = newTimer % 60;
      showTime->setText(QString().sprintf("%02d:%02d", a, b));
    } else {
      int a = newTimer / 60;
      int b = newTimer % 60;
      
      //showTime->setText(QString().sprintf("%02d%s%02d%s", a, "ʹ", b, "ʺ"));
      showTime->setText(QString().sprintf("%02d%s%02d%s", a, "'", b, "\""));
    }
    oldTimer = newTimer;
  }
}

void KModemLightsApplet::dial()
{
  if (isConnected()) {
    int ans = KMessageBox::questionYesNo(this, 
					 i18n("You are currently connected.\n"
					      "Do you want to disconnect?"));
    if (ans == KMessageBox::Yes)
      system(disconnectCommand.ascii());
  } else {
    int ans = KMessageBox::questionYesNo(this, i18n("Do you want to connect?"));
    if (ans == KMessageBox::Yes)
    system(connectCommand.ascii());
  }
}

void KModemLightsApplet::about()
{
  if(!aboutData) {
    aboutData = new KAboutData("kmodemlights", 
			       I18N_NOOP("KModemlights"), VERSION,
			       I18N_NOOP("Modem Lights Applet for KDE.\n\n"
					 "Shows status of a modem "// or ISDN "
					 "dialup connection"),
			       KAboutData::License_GPL, 
			       "(c) 2004, M G Berberich", 0,
			       "http://www.forwiss.uni-passau.de/~berberic/"
			       "Linux/kmodemlights/");
    aboutData->addAuthor("M G Berberich",
			 I18N_NOOP("KDE-Panelapplet"), 
			 "berberic@fmi.uni-passau.de",
			 "http://www.forwiss.uni-passau.de/~berberic/");
    aboutData->addAuthor("John Ellis",
			 I18N_NOOP("modemcode (from modemlights)"),
			 "johne@bellatlantic.net");
    aboutData->setTranslator(I18N_NOOP("_: NAME OF TRANSLATORS\\nYour names"),
			     I18N_NOOP("_: EMAIL OF TRANSLATORS\\nYour emails"));
  }
  
  KAboutApplication dialog(aboutData);
  dialog.exec();
}

void KModemLightsApplet::preferences()
{
  KModemLightsDlg dlg(this, "moondlg", this);
  if (dlg.exec() == KModemLightsDlg::Accepted) {
    // "write" config
    config()->setGroup("General");
    config()->writePathEntry("Lockfile", dlg.lockFile());
    config()->writeEntry("Devicename",  dlg.deviceName());
    config()->writeEntry("Connect", dlg.connectCommand());
    config()->writeEntry("Disconnect", dlg.disconnectCommand());
    config()->writeEntry("Update frequency", dlg.updateFrequency());

    // "write" colors
    config()->writeEntry("Rx on color", dlg.rxOnColor());
    config()->writeEntry("Rx off color", dlg.rxOffColor());
    config()->writeEntry("Rx graph color", dlg.rxGraphColor());
    config()->writeEntry("Tx on color", dlg.txOnColor());
    config()->writeEntry("Tx off color", dlg.txOffColor());
    config()->writeEntry("Tx graph color", dlg.txGraphColor());
    config()->writeEntry("Await color", dlg.statusAwaitColor());
    config()->writeEntry("On color", dlg.statusOnColor());
    config()->sync();

    readConfig();
  }
}

void KModemLightsApplet::readConfig()
{
  config()->setGroup("General");

  // read config
  lockFile = config()->readPathEntry("Lockfile", "/var/lock/LCK..ttyS1");
  deviceName = config()->readEntry("Devicename", "ppp0");
  connectCommand = config()->readEntry("Connect", "pon");
  disconnectCommand = config()->readEntry("Disconnect", "poff");
  updateFrequency = config()->readNumEntry("Update frequency", 5);

  // default colors
  static const QColor CGreen(0x00FF00);
  static const QColor CRed(0xFF0000);
  static const QColor CYellow(0xFFFF00);
  static const QColor CDarkGreen(0x004400);
  static const QColor CDarkRed(0x440000);

  // read colors
  lightRx->setColors(config()->readColorEntry("Rx on color", &CGreen),
		     config()->readColorEntry("Rx off color", &CDarkGreen));

  lightTx->setColors(config()->readColorEntry("Tx on color", &CRed),
		     config()->readColorEntry("Tx off color", &CDarkRed));

  graph->setColors(config()->readColorEntry("Rx graph color", &CGreen),
		   config()->readColorEntry("Tx graph color", &CRed));
  
  button->setColor(StatusButton::Wait,
		   config()->readColorEntry("Await color", &CYellow));
  button->setColor(StatusButton::On,
		   config()->readColorEntry("On color", &CGreen));
}
