/*
 * (c) 2004 M G Berberich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file called COPYING; if not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if_ppp.h>

#include "modem.h"

Modem::Modem()
  : DeviceName(0), LockFile(0)
{
  socket = ::socket(AF_INET, SOCK_DGRAM, 0);
}

Modem::Modem(char *device, char *lock)
  : DeviceName(0), LockFile(0)
{
  DeviceName = strdup(device);
  LockFile = strdup(lock);
  socket = ::socket(AF_INET, SOCK_DGRAM, 0);
}

Modem::~Modem()
{
  free(DeviceName);
  free(LockFile);
  close(socket);
}

bool Modem::isConnected() const
{
  FILE *f = 0;
  char buf[64];
  pid_t pid = -1;

  if (!LockFile) return false;

  f = fopen(LockFile, "r");
  
  if (!f) return false;
  
  if (fgets(buf, sizeof(buf), f) == 0) {
    fclose(f);
    return false;
  }
  fclose(f);
  
  pid = (pid_t)strtol(buf, NULL, 10);
  if (pid < 1 || (kill(pid, 0) == -1 && errno != EPERM)) 
    return false;
  
  return true;
}

bool Modem::getStats(unsigned int &in, unsigned int &out) const
{
  struct ifreq ifreq;
  struct ppp_stats stats;

  if (!DeviceName) goto FAIL;
  
  memset(&ifreq, 0, sizeof(ifreq));
  strncpy(ifreq.ifr_name, DeviceName, IFNAMSIZ);
  ifreq.ifr_name[IFNAMSIZ-1] = '\0';
  ifreq.ifr_ifru.ifru_data = (caddr_t)&stats;
  
#ifdef SIOCGPPPSTATS
  if (ioctl(socket, SIOCGPPPSTATS, (caddr_t)&ifreq) < 0) {
#else
  if (true) {
#endif
    /* failure means ppp is not up */
    goto FAIL;
  } else {
    in = stats.p.ppp_ibytes;
    out = stats.p.ppp_obytes;
    return true;
  }

  FAIL:
  in = out = 0;
  return false;   
}

int Modem::getConnectTime(bool recalc)
{
  static time_t start_time = (time_t)0;

  if (!LockFile) return -1;

  if (recalc) {
    struct stat st;
    if (stat(LockFile, &st) == 0)
      start_time = st.st_mtime;
    else
      start_time = (time_t)0;
  }

  if (start_time != (time_t)0)
    return (int)(time(0) - start_time);
  else
    return -1;
}
  
void  Modem::setDeviceName(const char *deviceName)
{
  if(DeviceName) free(DeviceName);
  DeviceName = strdup(deviceName);
}

void  Modem::setLockFile(const char *lockFile)
{
  if(LockFile) free(LockFile);
  LockFile = strdup(lockFile);
}
