/*
 * (c) 2004 M G Berberich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file called COPYING; if not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <malloc.h>
#include <math.h>

#include <qwidget.h>
#include <qpainter.h>
#include <qevent.h>

#include "graph.moc"

#define MAX(a, b) (a)>(b) ? (a) : (b)

Graph::Graph(QWidget *parent, const char *name)
  : QWidget(parent, name),
    dataSamples(0), dataSize(0), dataPtr(0), 
    dataMax(0), dataMaxSize(0), dataMaxPtr(0), 
    colorA(Qt::green), colorB(Qt::red)
{
  setup();
}

Graph::Graph(const QColor cA, const QColor cB, QWidget *parent, const char *name)
  : QWidget(parent, name),
    dataSamples(0), dataSize(0), dataPtr(0), 
    dataMax(0), dataMaxSize(0), dataMaxPtr(0), 
    colorA(cA), colorB(cB)
{
  setup();
}

inline void Graph::setup()
{
  setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
  setPaletteBackgroundColor(Qt::black);
}

Graph::~Graph()
{
  if(dataSamples) {
    free(dataSamples);
    free(dataMax);
  }
}

void Graph::addValues(unsigned int a, unsigned int b)
{
  if(!dataSize) return;
  
  // update the data
  if(dataPtr >= dataSize) { // storage is full
    dataPtr = dataSize;
    memmove(dataSamples, dataSamples+1, (dataSize-1)*sizeof(*dataSamples));
    dataSamples[dataPtr-1].a = a;
    dataSamples[dataPtr-1].b = b;
  } else { // not yet full
    dataSamples[dataPtr].a = a;
    dataSamples[dataPtr].b = b;
    ++dataPtr;
  }

  // update dataMax
  if(dataMaxPtr >= dataMaxSize) { // max is full
    dataMaxPtr = dataMaxSize;
    memmove(dataMax, dataMax+1, (dataMaxSize-1)*sizeof(*dataMax));
    dataMax[dataMaxPtr-1] = MAX(a, b);
  } else { // not yet full
    dataMax[dataMaxPtr] = MAX(a, b);
    ++dataMaxPtr;
  }

  // set the maxValue of the display
  maxValue = 0;
  for(unsigned int i=0; i<dataMaxPtr; ++i) {
    if (dataMax[i] > maxValue)
      maxValue = dataMax[i];
  }
  repaint();
}

void Graph::resizeEvent(QResizeEvent *event)
{
  dataSize = width();
  dataSamples = static_cast<Samples *>(realloc(dataSamples, 
					      sizeof(*dataSamples)*dataSize));
  if (dataPtr > dataSize) dataPtr = dataSize;

  dataMaxSize = 4*dataSize;
  dataMax = static_cast<unsigned int *>(realloc(dataMax, 
						sizeof(*dataMax)*dataMaxSize));
  if (dataMaxPtr > dataMaxSize) dataMaxPtr = dataMaxSize;

  QWidget::resizeEvent(event);
}

void Graph::paintEvent(QPaintEvent *)
{
  if(!maxValue) return;

  const int h = height();
  const double mh = double(h)/double(maxValue);

  QPainter p(this);
  p.setPen(colorA);
  for(unsigned int i=0; i<dataPtr; ++i) {
    int v = int(dataSamples[i].a*mh);
    if(v)
      p.drawLine(i, h-1, i, h-v);
  }
  p.setPen(colorB);
  for(unsigned int i=0; i<dataPtr; ++i) {
    int v = int(dataSamples[i].b*mh);
    if (v) 
      p.drawLine(i, h-1, i, h-v);
  }
  p.flush();
}
