/*
 *   kmodemlights - a moon phase indicator
 *   Copyright (C) 1998  Stephan Kulow
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <qspinbox.h>
#include <qlayout.h>
#include <qgrid.h>
#include <qwhatsthis.h>
#include <qpushbutton.h>
#include <qlineedit.h> 
#include <qgroupbox.h>

#include <klocale.h>
#include <kapplication.h>
#include <ktabwidget.h>
#include <kpushbutton.h>
#include <kfontdialog.h>
//
#include <kcolorbutton.h> 

#include "kmodemlights.h"
#include "light.h"
#include "graph.h"
#include "statusbutton.h"
#include "kmodemlightsdlg.moc"

KModemLightsDlg::KModemLightsDlg(QWidget *parent, const char *name,
				 KModemLightsApplet *ap)
  : KDialogBase(parent, name, true, i18n("Change Settings"),
		Ok|Cancel|Help),
    applet(ap)
{
  KTabWidget *page = new KTabWidget(this);
  page->setMargin(marginHint());
  setMainWidget(page);

  page->addTab(generateDevice(), i18n("Devicesettings"));
  page->addTab(generateColor(), i18n("Colors"));

  connect(this, SIGNAL(helpClicked()), SLOT(help()));
}

QWidget *KModemLightsDlg::generateDevice()
{
  QWidget *tab = new QWidget();
  QVBoxLayout *vbox = new QVBoxLayout(tab);
  QGrid *grid = new QGrid(2, tab);
  grid->setSpacing(spacingHint());
  vbox->add(grid);

  // Frequency
  QLabel *label = new QLabel(i18n("Update Frequency:"), grid, "caption" );
  frequencyWidget = new QSpinBox(1, 5, 1, grid);
  frequencyWidget->setValue(applet->updateFrequency);
  frequencyWidget->setSuffix(" Hz");
  QString text = i18n("<b>This is the frequency the applet is updated.</b>"
		      "<p>A value of 3 means it is updated 3 times a second."
		      "The default value is 5. On extremly slow systems "
		      "you might like to reduce the frequency to lower "
		      "the system-load, mostly the default is O.K.</p>");
  QWhatsThis::add(label, text);
  QWhatsThis::add(frequencyWidget, text);

  // Device Name
  label = new QLabel(i18n("Device:"), grid);
  text = i18n("<b>This is the name of the device.</b>"
	      "<p>P.e. 'ppp0' for (the first) PPP device.</p>");
  QWhatsThis::add(label, text);  
  deviceWidget = new QLineEdit(applet->modem.deviceName(), grid);
  QWhatsThis::add(deviceWidget, text);
  
  // Lockfile Name
  label = new QLabel(i18n("Lockfile:"), grid);
  text = i18n("<b>This is the path to the file, that is used to lock"
	      "the device.</b>"
	      "<p>On a standard linux-system this is "
	      "<nobr>'/var/lock/LCK..ttyS1'</nobr> for a modem "
	      "on <nobr>'/dev/ttyS1'</nobr>.</p>");
  QWhatsThis::add(label, text);
  lockFileWidget = new QLineEdit(applet->modem.lockFile(), grid);
  QWhatsThis::add(lockFileWidget, text);

  // Connect command
  label = new QLabel(i18n("Connect:"), grid);
  text = i18n("<b>This is the command that is executed to go online.</b>"
	      "<p>It is different on different distributions and systems.</p>" 
	      "<p>For a connection configured with kppp it is "
	      "<nobr>'kppp -c &lt;provider&gt;'</nobr>.</p>");
  QWhatsThis::add(label, text);
  connectWidget = new QLineEdit(applet->connectCommand, grid);
  QWhatsThis::add(connectWidget, text);

  // Disonnect command
  label = new QLabel(i18n("Disconnect:"), grid);
  text = i18n("<b>This is the command that is executed to go offline.</b>"
	      "<p>It is different on different distributions and systems.</p>"
	      "<p>For a connection configured with kppp it is "
	      "<nobr>'kppp -k -q'<nobr>.</p>");
  QWhatsThis::add(label, text);
  disconnectWidget = new QLineEdit(applet->disconnectCommand, grid);
  QWhatsThis::add(disconnectWidget, text);

  // Hint to use What's this help
  vbox->addStretch();
  label = new QLabel(i18n("<i>There is help available for all fields on this "
			  "tab, that can be shown with Shift-F1.</i>"), tab);
  label->setTextFormat(Qt::RichText); // does not break the text otherwise
  vbox->addWidget(label);

  return tab;
}

static QLabel *newCaption(QString s, QWidget *w)
{
  QLabel *l = new QLabel(s, w);
  l->setAlignment(Qt::AlignHCenter|Qt::AlignVCenter);

  return l;
}

QWidget *KModemLightsDlg::generateColor()
{
  QWidget *tab = new QWidget();
  QVBoxLayout *vbox = new QVBoxLayout(tab);
  vbox->setSpacing(spacingHint());

  QGroupBox *group = new QGroupBox(2, Qt::Vertical, i18n("Receive Data"), tab);
  vbox->add(group);
  rxOn = new KColorButton(group);
  QLabel *label = newCaption(i18n("On"), group);
  rxOff = new KColorButton(group);
  label = newCaption(i18n("Off"), group);
  rxGraph = new KColorButton(group);
  label = newCaption(i18n("Graph"), group);

  group = new QGroupBox(2, Qt::Vertical, i18n("Send Data"), tab);
  vbox->add(group);
  txOn = new KColorButton(group);
  label = newCaption(i18n("On"), group);
  txOff = new KColorButton(group);
  label = newCaption(i18n("Off"), group);
  txGraph = new KColorButton(group);
  label = newCaption(i18n("Graph"), group);

  group = new QGroupBox(2, Qt::Vertical, i18n("Connection Status"), tab);
  vbox->add(group);
  statusAwait = new KColorButton(group);
  label = newCaption(i18n("Await Connection"), group);
  statusOn = new KColorButton(group);
  label = newCaption(i18n("Connected"), group);

  vbox->addStretch();  

  // setup colors in widgets
  rxOn->setColor(applet->lightRx->getColorOn());
  rxOff->setColor(applet->lightRx->getColorOff());
  rxGraph->setColor(applet->graph->getColorA());

  txOn->setColor(applet->lightTx->getColorOn());
  txOff->setColor(applet->lightTx->getColorOff());
  txGraph->setColor(applet->graph->getColorB());

  statusAwait->setColor(applet->button->getColor(StatusButton::Wait));
  statusOn->setColor(applet->button->getColor(StatusButton::On));

  return tab;
}

void KModemLightsDlg::help() {
  kapp->invokeHelp(QString::fromLatin1("config"));
}
