/*
 * (c) 2004 M G Berberich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file called COPYING; if not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <unistd.h>
#include <wait.h>

#include <qlayout.h>

#include "kpanelapplet.h"
#include <kiconloader.h>
#include <klocale.h>
#include <kglobalsettings.h>
#include <kmessagebox.h>
#include <kaboutapplication.h>
#include <kpopupmenu.h>

#include "statusbutton.h"
#include "light.h"
#include "graph.h"
#include "speciallabels.h"
#include "modem.h"
#include "kmodemlightsdlg.h"
#include "kmodemlights.moc"

extern "C" {
  KPanelApplet *init(QWidget *parent, const QString& configFile)
  {
    KGlobal::locale()->insertCatalogue("kmodemlights");
    return new KModemLightsApplet(configFile, KPanelApplet::Normal,
				  KPanelApplet::About|KPanelApplet::Preferences,
				  parent, "kmodemlightapplet");
  }
}

extern char **environ;

static int system_nowait(const char *command)
{
  if (command == 0)
    return 1;
  
  /* go in "daemon mode" */
  pid_t pid = fork();
  if (pid < 0) return -1;	/* fork failed */
  if (pid == 0) {    		/* we are the child */
    pid_t cpid = fork();	/* fork the "daemon" */
    if (cpid < 0) exit(-1);	/* fork failed */
    if (cpid > 0) exit(0);	/* exit parent, to satify waiting process */
    setsid();			/* found new processgroup */
    const char * argv[4];
    argv[0] = "sh";
    argv[1] = "-c";
    argv[2] = command;
    argv[3] = 0;
    execve("/bin/sh", (char *const *)argv, environ);
    exit(127);
  }
  waitpid(pid, 0, 0);
  return 0;
}

KModemLightsApplet::KModemLightsApplet(const QString& configFile, Type type, 
				       int actions, QWidget *parent, 
				       const char *name)
  : KPanelApplet(configFile, type, actions, parent, name),
    aboutData(0),
    timerID(0),
    timerEventCounter(0),
    lastTimeWasConnected(false),
    modemWasOn(false),
    oldRxBytes(-1)
{
  KGlobal::iconLoader()->addAppDir("kmodemlights");

  // The applett itself

  // Button
  button = new StatusButton(this);
  button->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
  QObject::connect(button, SIGNAL(clicked()), this, SLOT(dial()));

  // graph and two lights
  lightRx = new Light(QColor(0x00FF00), QColor(0x004400), this);
  lightRx->setBackgroundOrigin(AncestorOrigin);
  lightTx = new Light(QColor(0xFF0000), QColor(0x440000), this);
  lightTx->setBackgroundOrigin(AncestorOrigin);
  graph = new Graph(this);

  // textual Displays
  showRx = new TRLabel(this);
  showRx->setBackgroundOrigin(AncestorOrigin);
  showTime = new TRLabel(this);
  showTime->setBackgroundOrigin(AncestorOrigin);
  showTime->setTime(-1);

  // layouts
  page = new QBoxLayout(this, QBoxLayout::LeftToRight, 0, 1);
  page->setResizeMode(QLayout::FreeResize);
  page->addWidget(button);
  infoPart = new QBoxLayout(page, QBoxLayout::TopToBottom, 1);
  QHBoxLayout *graphPart = new QHBoxLayout(infoPart, 0);
  graphPart->addWidget(graph);
  QVBoxLayout *vbox1 = new QVBoxLayout(graphPart, 0);
  vbox1->addWidget(lightRx);
  vbox1->addWidget(lightTx);
  QVBoxLayout *textualPart = new QVBoxLayout(infoPart, 0);
  textualPart->addWidget(showRx);
  textualPart->addWidget(showTime);

  // popupMenu
  popup = new KPopupMenu();
  popup->insertTitle(SmallIcon("kmodemlighs"), i18n("kmodemlights"));
  popup->insertItem(i18n("&About kmodemlights..."), this,
		    SLOT(about()));
  popup->insertSeparator();
  popup->insertItem(SmallIcon("configure"), i18n("&Settings..."), this,
		    SLOT(preferences()));

  readConfig();
  
  timerID = startTimer(1000/updateFrequency);
}

KModemLightsApplet::~KModemLightsApplet()
{
  killTimer(timerID);
  if(aboutData) delete aboutData;
}

int KModemLightsApplet::widthForHeight(int height) const
{
  // two-row-layout or single-row-layout
  int widgetheight = height >= 42 ? (height-1)/4 : height/2;
  
  // button
  int b = button->sizeHint().width();

  // graph + lights
  int w1 = 22 + (widgetheight > 12 ? 12 : 9); 
 
  // take the configured KDE fixed font
  QFont font(KGlobalSettings::fixedFont());
  font.setPixelSize(widgetheight);
  // guess the maximum size by calulate the size of a string set to 
  // 4 digits and 2 characters (b,m,' or ")
  int w2 = (QFontMetrics(font).width("88m88m"));

  if (height >= 42) // two-row-layout 
    return b + page->spacing() + (w1>w2 ? w1 : w2);
  else  // single-row-layout
    return b + page->spacing() + w1 + infoPart->spacing() + w2;
}

static inline int fontHeightForWidth(int low, int up, int w, 
				     const QString s, QFont f)
{
  f.setPixelSize(up);
  int upWidth = QFontMetrics(f).width(s);
  if (upWidth < w) return up;
  f.setPixelSize(low);
  int lowWidth = QFontMetrics(f).width(s);
  if (lowWidth > w) return low;
  do {
    if (up == low || up == low+1) break;
    int mid = (up+low)/2;
    f.setPixelSize(mid);
    int midWidth = QFontMetrics(f).width(s);
    if (midWidth < w) {
      lowWidth = midWidth;
      low = mid;
    } else {
      upWidth = midWidth;
      up = mid;
    }
  } while(1);
  return  low;
}

int KModemLightsApplet::heightForWidth(int width) const
{
  // button
  int b = button->sizeHint().height();

  // lights + graph
  int h1 = 2*9;
  
  // take the configured KDE fixed font
  QFont font(KGlobalSettings::fixedFont());
  int h2 = 2 * fontHeightForWidth(4, 48, width, "88m88m", font);

  return b + page->spacing() + h1 + infoPart->spacing() + h2;
}

void KModemLightsApplet::resizeEvent(QResizeEvent *event)
{
  switch(position()) {
  case pTop:
  case pBottom:
    horizontalLayout(height());
    break;
  case pLeft:
  case pRight:
    verticalLayout(width());
    break;
  default:
    return;
  }
  QWidget::resizeEvent(event);
}

void KModemLightsApplet::horizontalLayout(int height)
{
  int widgetheight;
  if(page->direction() != QBoxLayout::LeftToRight)
    page->setDirection(QBoxLayout::LeftToRight);
  if (height >= 42) {	// two-row-layout
    widgetheight = height/4;
    if(infoPart->direction() != QBoxLayout::TopToBottom)
      infoPart->setDirection(QBoxLayout::TopToBottom);
  } else {		// single-row-layout
    widgetheight = height/2;
    if(infoPart->direction() != QBoxLayout::LeftToRight)
      infoPart->setDirection(QBoxLayout::LeftToRight); 
  }
      
  if(widgetheight > 12) {
    lightRx->setBig(true);
    lightTx->setBig(true);
  } else {
    lightRx->setBig(false);
    lightTx->setBig(false);
  }
      
  // take the configured KDE fixed font
  QFont font(KGlobalSettings::fixedFont());
  font.setPixelSize(widgetheight);
  //bmg  int textwidth = QFontMetrics(font).width("88m88m");
  // and set it on the text-widgets
  showRx->setFont(font);
  showRx->setFixedHeight(widgetheight);
  showTime->setFont(font);
  showTime->setFixedHeight(widgetheight);
}

void KModemLightsApplet::verticalLayout(int width)
{
  if(page->direction() != QBoxLayout::TopToBottom)
    page->setDirection(QBoxLayout::TopToBottom);
  if(infoPart->direction() != QBoxLayout::TopToBottom)
    infoPart->setDirection(QBoxLayout::TopToBottom);
  lightRx->setBig(false);
  lightTx->setBig(false);
      
  // take the configured KDE fixed font
  QFont font(KGlobalSettings::fixedFont());
  int fontheight = fontHeightForWidth(4, 48, width, "88m88m", font);
  font.setPixelSize(fontheight);
  // and set it on the text-widgets
  showRx->setFont(font);
  showRx->setFixedHeight(fontheight);
  showTime->setFont(font);
  showTime->setFixedHeight(fontheight);
}

void KModemLightsApplet::mousePressEvent(QMouseEvent *e)
{
  if (e->button() == RightButton) {
    popup->popup(mapToGlobal(e->pos()));
    popup->exec();
  }
}

void KModemLightsApplet::timerEvent(QTimerEvent *)
{
  unsigned int rx, tx;

  timerEventCounter++;

  if (modem.isConnected()) {
    bool statsRet = modem.getStats(rx, tx);

    // Timer
    if (!lastTimeWasConnected) {
      modem.getConnectTime(true); // reset start time
      lastTimeWasConnected = true;
    }
    updateTimer();

    // button and lights
    if (!statsRet || (rx == 0 && tx == 0)) {
      button->setStatus(StatusButton::Wait);
      oldRx = oldTx = 0;
      updateLights(false, false);
    } else {
      button->setStatus(StatusButton::On);
      updateLights((rx > oldRx), (tx > oldTx));
      oldRx = rx;
      oldTx = tx;
    }

    // (all 1 sec)
    if (timerEventCounter % updateFrequency == 0) {
      if (statsRet)
	updateRx(rx);

      // (all 2 sec)
      if (timerEventCounter % (updateFrequency * 2) == 0) {	
	if (!modemWasOn) {
	  graphRx = rx; // reset graphRx/graphTx
	  graphTx = tx;
	  modemWasOn = true;
	}
	
	updateGraph((rx-graphRx)/2, (tx-graphTx)/2); // 2sec!
	graphRx = rx;
	graphTx = tx;
      }
    }
  } else {
    button->setStatus(StatusButton::Off);
    updateRx(-1);
    updateLights(false, false);
    
    // (all 2 sec)
    if (timerEventCounter % (updateFrequency * 2) == 0) {
      updateGraph(0, 0);
    }
    if (modemWasOn) modemWasOn = false;
    if (lastTimeWasConnected) lastTimeWasConnected = false;
  }
}

void KModemLightsApplet::updateLights(bool rx, bool tx)
{
  lightRx->setStatus(rx);
  lightTx->setStatus(tx);
}

void KModemLightsApplet::updateGraph(unsigned int rx, unsigned int tx)
{
  graph->addValues(rx, tx);
}

void KModemLightsApplet::updateRx(int rxBytes)
{
  // reset
  if (rxBytes == -1) {
    oldRxBytes = -1;
    showRx->setText("");
    return;
  }
  // init
  if (oldRxBytes == -1) {
    oldRxBytes = rxBytes;
    return;
  }

  if (oldRxBytes > rxBytes) {
    oldRxBytes = rxBytes;
    return;
  }

  int rx = rxBytes - oldRxBytes;
  oldRxBytes = rxBytes;

  showRx->setBytes(rx);
}

void  KModemLightsApplet::updateTimer()
{
  int newTimer = modem.getConnectTime(false);
  showTime->setTime(newTimer);
}

void KModemLightsApplet::dial()
{
  int ans = KMessageBox::Yes;
  
  if (modem.isConnected()) {
    if (showRequesters) {
      ans = KMessageBox::questionYesNo(this, 
				       i18n("You are currently connected.\n"
					    "Do you want to disconnect?"));
    }
    if (ans == KMessageBox::Yes)
      system_nowait(disconnectCommand.ascii());
  } else {
    if (showRequesters) {
      ans = KMessageBox::questionYesNo(this, i18n("Do you want to connect?"));
    }
    if (ans == KMessageBox::Yes)
      system_nowait(connectCommand.ascii());
  }
}

void KModemLightsApplet::about()
{
  if(!aboutData) {
    aboutData = new KAboutData("kmodemlights", 
			       I18N_NOOP("KModemlights"), VERSION,
			       I18N_NOOP("Modem Lights Applet for KDE.\n\n"
					 "Shows status of a modem "// or ISDN "
					 "dialup connection"),
			       KAboutData::License_GPL, 
			       "(c) 2004, M G Berberich", 0,
			       "http://www.forwiss.uni-passau.de/~berberic/"
			       "Linux/kmodemlights/");
    aboutData->addAuthor("M G Berberich",
			 I18N_NOOP("KDE-Panelapplet"), 
			 "berberic@fmi.uni-passau.de",
			 "http://www.forwiss.uni-passau.de/~berberic/");
    aboutData->addAuthor("John Ellis",
			 I18N_NOOP("modemcode (from modemlights)"),
			 "johne@bellatlantic.net");
    aboutData->setTranslator(I18N_NOOP("_: NAME OF TRANSLATORS\\nYour names"),
			     I18N_NOOP("_: EMAIL OF TRANSLATORS\\nYour emails"));
  }
  
  KAboutApplication dialog(aboutData);
  dialog.exec();
}

void KModemLightsApplet::preferences()
{
  KModemLightsDlg dlg(this);
  if (dlg.exec() == KModemLightsDlg::Accepted) {
    config()->setGroup("General");

    // "write" device
    config()->writePathEntry("Lockfile", dlg.lockFile());
    config()->writeEntry("Devicename",  dlg.deviceName());
    config()->writeEntry("Connect", dlg.connectCommand());
    config()->writeEntry("Disconnect", dlg.disconnectCommand());

    // write general
    config()->writeEntry("Update frequency", dlg.updateFrequency());
    config()->writeEntry("Show requesters", dlg.showRequesters());
    config()->writeEntry("Graph automax", dlg.graphAutoMax());
    config()->writeEntry("Graph maxvalue", dlg.graphMaxValue());

    // "write" colors
    config()->writeEntry("Rx on color", dlg.rxOnColor());
    config()->writeEntry("Rx off color", dlg.rxOffColor());
    config()->writeEntry("Rx graph color", dlg.rxGraphColor());
    config()->writeEntry("Tx on color", dlg.txOnColor());
    config()->writeEntry("Tx off color", dlg.txOffColor());
    config()->writeEntry("Tx graph color", dlg.txGraphColor());
    config()->writeEntry("Graph background", dlg.graphBgColor());
    config()->writeEntry("Await color", dlg.statusAwaitColor());
    config()->writeEntry("On color", dlg.statusOnColor());
    config()->sync();

    readConfig();
  }
}

void KModemLightsApplet::readConfig()
{
  config()->setGroup("General");

  // read device
  QString lockFile = config()->readPathEntry("Lockfile", "/var/lock/LCK..ttyS1");
  modem.setLockFile(lockFile.ascii());
  QString deviceName = config()->readEntry("Devicename", "ppp0");
  modem.setDeviceName(deviceName.ascii());
  connectCommand = config()->readEntry("Connect", "pon");
  disconnectCommand = config()->readEntry("Disconnect", "poff");

  // read general
  updateFrequency = config()->readNumEntry("Update frequency", 5);
  showRequesters = config()->readBoolEntry("Show requesters", true);
  graphMaxValue = config()->readNumEntry("Graph maxvalue", 10000);
  bool graphAutoMax = config()->readBoolEntry("Graph automax", true);
  if (graphAutoMax) graph->unsetMaxValue();
  else
    graph->setMaxValue(graphMaxValue);

  // default colors
  static const QColor CGreen(0x00FF00);
  static const QColor CRed(0xFF0000);
  static const QColor CYellow(0xFFFF00);
  static const QColor CDarkGreen(0x004400);
  static const QColor CDarkRed(0x440000);

  // read colors
  lightRx->setColors(config()->readColorEntry("Rx on color", &CGreen),
		     config()->readColorEntry("Rx off color", &CDarkGreen));

  lightTx->setColors(config()->readColorEntry("Tx on color", &CRed),
		     config()->readColorEntry("Tx off color", &CDarkRed));

  graph->setColorA(config()->readColorEntry("Rx graph color", &CGreen));
  graph->setColorB(config()->readColorEntry("Tx graph color", &CRed));
  graph->setColorBg(config()->readColorEntry("Graph background", &Qt::black));
  
  button->setColor(StatusButton::Wait,
		   config()->readColorEntry("Await color", &CYellow));
  button->setColor(StatusButton::On,
		   config()->readColorEntry("On color", &CGreen));

}
