/*
 * (c) 2004 M G Berberich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program in a file called COPYING; if not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <qwidget.h>
#include <qpainter.h>

#include "light.moc"

#include "light.data"

// define static members
QPixmap Light::lightPixmapS, Light::lightMaskS;  
QPixmap Light::lightPixmapL, Light::lightMaskL;

Light::Light(const QColor &cOn, const QColor &cOff, 
	     QWidget *parent, const char *name)
  : QWidget(parent, name),
    lightStatus(false), bigLights(false),
    colorOn(cOn), colorOff(cOff), offx(0), offy(0)
    
{
  if(lightPixmapS.isNull()) {
    lightPixmapS.loadFromData(light_9, light_9_size);
    lightMaskS.loadFromData(light_mask_9, light_mask_9_size);
    lightPixmapL.loadFromData(light_12, light_12_size);
    lightMaskL.loadFromData(light_mask_12, light_mask_12_size);
  }

  generateIcon(on, colorOn);
  generateIcon(off, colorOff);
  
  setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
}

void Light::paintEvent(QPaintEvent *)
{
  QPainter paint(this);
  paint.drawPixmap(offx, offy, lightStatus ? on : off);
}

void Light::resizeEvent(QResizeEvent *e)
{
  offx = (width() - on.width())/2;
  offy = (height() - on.height())/2;

  QWidget::resizeEvent(e);
}

QSize Light::sizeHint() const
{
  return on.size();
}

QSize Light::minimumSizeHint() const
{
  return on.size();
}

void Light::setStatus(bool s)
{
  if (s == lightStatus)
    return;

  lightStatus = s;
  repaint();
}

void Light::setColors(const QColor &cOn, const QColor &cOff)
{
  if (colorOn != cOn) {
    colorOn = cOn;
    generateIcon(on, colorOn);
    if (lightStatus)
      update();
  }
  if (colorOff != cOff) {
    colorOff = cOff;
    generateIcon(off, colorOff);
    if (!lightStatus)
      update();
  }
}

void Light::setBig(bool b)
{
  if (b == bigLights)
    return;

  bigLights = b;
  generateIcon(on, colorOn);
  generateIcon(off, colorOff);
  update();
}

void Light::generateIcon(QPixmap &px, QColor &col)
{
  px = bigLights ? lightMaskL : lightMaskS;
  QPainter p(&px);
  p.fillRect(0, 0, on.width()-1, on.height()-1, col);
  p.drawPixmap(0, 0, bigLights ? lightPixmapL : lightPixmapS);
  p.end();
}


