/*
 * ppplog.c -- format log of ppp-connect-logs
 *
 * (c) 2001 M G Berberich <berberic@fmi.uni-passau.de> 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  $Id: ppplog.c,v 1.9 2001/03/13 17:27:34 berberic Exp berberic $ 
 */


/* needed for strptime prototype */
#define _GNU_SOURCE 

#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <malloc.h>
#include <getopt.h>

#ifndef TRUE
#define TRUE            1
#define FALSE           0
#endif

#ifndef VERSION
  #define VERSION "UNDEFINED"
#endif

/**********************************************************************/

struct sumfield {
  char *provider;
  unsigned long sum, insum, outsum;
};

struct vector {
  unsigned int size;
  unsigned int memsize;
  struct sumfield *data;
};

static struct vector *init_vector(struct vector *v, int a)
{
  int i;
  v->size = 0;
  v->data = malloc(sizeof(struct sumfield)*a);
  if (!v->data)
    return 0;
  v->memsize = a;
  for (i=0; i<a; i++) {
    v->data[i].provider = 0;
    v->data[i].sum = 0;
  }
  return v;
}

static int add_to_vector(struct vector *v, char *p, unsigned long a, 
			 unsigned long in, unsigned long out)
{
  int i;

  /* is the provider already in the list? */
  for (i=0; i<v->size; i++) {
    if (!strcmp(v->data[i].provider, p)) {
      v->data[i].sum += a;
      v->data[i].insum += in;
      v->data[i].outsum += out;
      return TRUE;
    }
  }

  /* add the provider to the list */
  if (v->size < v->memsize) {
    v->data[v->size].provider = strdup(p);
    v->data[v->size].sum = a;
    v->data[i].insum = in;
    v->data[i].outsum = out;
    v->size++;
  } else { /* we must grow the list */
    struct sumfield *data2 = v->data;
    v->data = malloc(sizeof(struct sumfield)*2*v->memsize);
    if (!v->data)
      return FALSE;
    for (i=0; i<v->memsize; i++) {
      v->data[i].provider = 0;
      v->data[i].sum = v->data[i].insum = v->data[i].outsum = 0;

    }
    memcpy(v->data, data2, sizeof(struct sumfield)*v->memsize);
    v->memsize *= 2;
    free(data2);
    v->data[v->size].provider = strdup(p);
    v->data[v->size].sum = a;
    v->data[i].insum = in;
    v->data[i].outsum = out;
    v->size++;
  }
  return TRUE;
}

/**********************************************************************/

struct _unittab {
  unsigned long long amount;
  char unit;
};

struct _unittab unittab[] = {
  {1024ULL, 'K'},
  {1048576ULL, 'M'},
  {1073741824ULL, 'G'},
  {1099511627776ULL, 'T'},
  {1125899906842624ULL, 'P'},
};

static void number_to_unit(unsigned long long a, double *r, char *u)
{
  int i;
  for (i=4; i>=0; i--) {
    if (a>unittab[i].amount) {
      *u = unittab[i].unit;
      *r = (double)a/unittab[i].amount;
      return;
    }
  }
  *u = ' ';
  *r = (int)a;
}

/**********************************************************************/

static void sec_to_hms(int t, int *h, int *m, int *s)
{
  *s = t%60;
  t /= 60;
  *m = t%60;
  t /= 60;
  *h = t%60;
}

static int sec_to_string(int t, char *s, int l)
{
  int h, min, sec;

  sec_to_hms(t, &h, &min, &sec);
  if (h)
    return snprintf(s, l, "%2d:%02d:%02d", h, min, sec);
  else
    return snprintf(s, l, "%2d:%02d", min, sec);
}

/**********************************************************************/

static void print_transfer(unsigned long in, unsigned long out)
{
  double i, o;
  char ui, uo;
  
  number_to_unit(in, &i, &ui);
  number_to_unit(out, &o, &uo);
  printf("  %4.3g%c  %4.3g%c", i, ui, o, uo);
}

/**********************************************************************/

static const char usage[] =
"ppplog version " VERSION "\n"
"usage: ppplog [OPTION]\n"
"   or: ppplog [OPTION] month-specification\n"
"Print ppplog for a specified or this month in human-readable form\n"
"month-specification := yyyy-mm | yyyy mm\n"
"\n"
"  -f, --file <logfile>  use data from file\n"
"  -t, --transfer        print transfer-volume too (1K=1024)\n"
"  -V, --version         print version\n"
"      --help            this message\n";

/**********************************************************************/

int main(int argc, char **argv)
{
  FILE *data = 0;
  char date[11], olddate[11]="", endtime[9], zone[5];
  char provider[21], interface[21];
  unsigned long connect, in, out;
  char s1[20], *s2, s3[30];
  char *filename = 0;
  struct tm *start, end;
  char starts[9];
  time_t t;
  int i, res, transfer = FALSE;
  unsigned long sum = 0, insum = 0, outsum = 0;
  struct vector provider_sum;
  const struct option longopts[] = {
    { "file",     required_argument, 0, 'f' },
    { "transfer", no_argument,       0, 't' },
    { "help",     no_argument,       0, '?' },
    { "version",  no_argument,       0, 'V' },
    { 0, 0, 0, 0 } };

  for(;;) {
    int c = getopt_long(argc, argv, "f:t?V", longopts, 0);
    if (c == -1)
      break;
    switch(c) {
    case 'f' :
      filename = optarg;
      break;
    case 't' :
      transfer = TRUE;
      break;
    case 'V' :
      puts("ppplog version " VERSION);
      return 0;
      break;
    case '?' :
      fprintf(stderr, usage);
      return 0;
      break;
    default:
	printf ("?? getopt returned character code 0%o ??\n", c);
	return 5;
    }
  }
  
  if (filename) {
    if (argv[optind]) {
      fprintf(stderr, "filename and month-specification"
	      "are mutual excluse!\n");
      return 4;
    }
  } else {
    if (argv[optind]) {
      filename = malloc(FILENAME_MAX+1);
      if (!filename) {
	fprintf(stderr, "can't allocate memory for filename?!\n");
	return 2;
      }
      if (argv[optind+1]) 
	snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%s-%s", 
		argv[optind], argv[optind+1]);
      else
	snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%s", argv[optind]);
    } else {
      time_t  t_t = time(0);
      struct tm *t_s = localtime (&t_t);
      char buffer[8];
      
      strftime (buffer, 8, "%Y-%m", t_s);
      
      filename = malloc(FILENAME_MAX+1);
      snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%s", buffer);
    }
  }
  
  if(!init_vector(&provider_sum, 5)) {
    fprintf(stderr, "Can't allocate vector ?!\n");
    return 2;
  }
  
  if (!(data = fopen(filename, "r"))) {
    fprintf(stderr, "Could not open file %s: %s\n", filename, strerror(errno));
    return 3;
  }
  
  printf("Date       Start       End          Provider    Connect");
  if (transfer)
    printf("     In    Out");
  printf("\n");

  while(8==(res = fscanf(data, "%10s %8s %4s %*s %20s %20s %ld %ld %ld\n", 
		  date, endtime, zone, provider, interface, 
		  &connect, &in, &out))) {

    /* nice print of connecttime */
    sec_to_string(connect, s1, 20);
    
    /* some magic to get the start-time of the connection */
    snprintf(s3, 30, "%s %s", date, endtime);
    memset(&end, 0, sizeof(struct tm));
    strptime(s3, "%Y-%m-%d %H:%M:%S", &end); /* Should not have used this :( */
    end.tm_zone = zone; /* there seems to be a bug in zone-handling */
    t = mktime(&end)-connect;
    start = localtime(&t);
    strftime(starts, 9, "%H:%M:%S", start);
    /* we need the date of the connection-start not of the connection-end */
    strftime(date, 11, "%Y-%m-%d", start);
    
    /* a new day ? */
    if (strcmp(date, olddate)) {
      strcpy(olddate, date);
      s2 = date;
    } else {
      s2 = "";
    }

    /* sum up the connect-times */
    if(!add_to_vector(&provider_sum, provider, connect, in ,out)) {
      fprintf(stderr, "Can't re-allocate vector ?!\n");
      return 2;
    }
    sum += connect;
    insum += in;
    outsum += out;

    /* print a line of the report */
    printf("%10s %8s -- %8s %12s %10s", s2, starts , endtime,  provider, s1);
    if (transfer) 
      print_transfer(in, out);
    printf("\n");
  }

  if (res != EOF) {
    fprintf(stderr, "error scanning file\n");
    return 1;
  }

  /* print sums of the providers */
  if (transfer)
    printf("-------------------------------------------------------"
	   "--------------\n");
  else
    printf("-------------------------------------------------------\n");
  for(i=0; i<provider_sum.size; i++) {
    sec_to_string(provider_sum.data[i].sum, s1, 20);
    printf("Sum                             %12s %10s", 
	   provider_sum.data[i].provider, s1);
    if (transfer) 
      print_transfer(provider_sum.data[i].insum, provider_sum.data[i].outsum);
    printf("\n");
      
  }

  /* print total sum */
  if (transfer)
    printf("======================================================="
	   "==============\n");
  else
    printf("=======================================================\n");

  sec_to_string(sum, s1, 20);
  printf("Sum                             %12s %10s", "total", s1);

  if (transfer) 
    print_transfer(insum, outsum);
  printf("\n");
  
  return 0;
}
