/*
 * ppplog.c -- format log of ppp-connect-logs
 *
 * (c) 2001 M G Berberich <berberic@fmi.uni-passau.de> 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this progggram; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  $Id: ppplog.c,v 1.15 2002/10/10 16:02:22 berberic Exp $ 
 */


/* needed for strptime prototype */
#define _GNU_SOURCE 

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <malloc.h>
#include <getopt.h>
#include <setjmp.h>
#include <ctype.h>

#include "ppplog.h"

#ifndef TRUE
#define TRUE            1
#define FALSE           0
#endif

#ifndef VERSION
  #define VERSION "UNDEFINED"
#endif

/**********************************************************************/

struct sumfield {
  char *provider;
  unsigned long sum, insum, outsum;
};

struct vector {
  unsigned int size;
  unsigned int memsize;
  struct sumfield *data;
};


struct entry {
  time_t t;
  char provider[21];
  char interface[21];
  unsigned long connect;
  unsigned long in;
  unsigned long out;
};

struct totals {
  struct vector provider_sum;
  unsigned long sum;
  unsigned long insum;
  unsigned long outsum;
};

/**********************************************************************/

static struct vector *init_vector(struct vector *v, int a)
{
  int i;
  v->size = 0;
  v->data = malloc(sizeof(struct sumfield)*a);
  if (!v->data)
    return 0;
  v->memsize = a;
  for (i=0; i<a; i++) {
    v->data[i].provider = 0;
    v->data[i].sum = 0;
  }
  return v;
}

static int add_to_vector(struct vector *v, const struct entry *e)
{
  int i;

  /* is the provider already in the list? */
  for (i=0; i<v->size; i++) {
    if (!strcmp(v->data[i].provider, e->provider)) {
      v->data[i].sum += e->connect;
      v->data[i].insum += e->in;
      v->data[i].outsum += e->out;
      return TRUE;
    }
  }

  /* add the provider to the list */
  if (v->size < v->memsize) {
    v->data[v->size].provider = strdup(e->provider);
    v->data[v->size].sum = e->connect;
    v->data[i].insum = e->in;
    v->data[i].outsum = e->out;
    v->size++;
  } else { /* we must grow the list */
    struct sumfield *data2 = v->data;
    v->data = malloc(sizeof(struct sumfield)*2*v->memsize);
    if (!v->data)
      return FALSE;
    for (i=0; i<v->memsize; i++) {
      v->data[i].provider = 0;
      v->data[i].sum = v->data[i].insum = v->data[i].outsum = 0;

    }
    memcpy(v->data, data2, sizeof(struct sumfield)*v->memsize);
    v->memsize *= 2;
    free(data2);
    v->data[v->size].provider = strdup(e->provider);
    v->data[v->size].sum = e->connect;
    v->data[i].insum = e->in;
    v->data[i].outsum = e->out;
    v->size++;
  }
  return TRUE;
}

/**********************************************************************/

struct _unittab {
  unsigned long long amount;
  char unit;
};

const struct _unittab unittab[] = {
  {1024ULL, 'K'},
  {1048576ULL, 'M'},
  {1073741824ULL, 'G'},
  {1099511627776ULL, 'T'},
  {1125899906842624ULL, 'P'},
};

static void number_to_unit(unsigned long long a, double *r, char *u)
{
  int i;
  for (i=4; i>=0; i--) {
    if (a>=unittab[i].amount) {
      *u = unittab[i].unit;
      *r = (double)a/unittab[i].amount;
      return;
    }
  }
  *u = ' ';
  *r = (int)a;
}

/**********************************************************************/

static void sec_to_hms(int t, int *h, int *m, int *s)
{
  *s = t%60;
  t /= 60;
  *m = t%60;
  t /= 60;
  *h = t;
}

static int sec_to_string(int t, char *s, int l)
{
  int h, min, sec;

  sec_to_hms(t, &h, &min, &sec);
  if (h)
    return snprintf(s, l, "%d:%02d:%02d", h, min, sec);
  else
    return snprintf(s, l, "%d:%02d", min, sec);
}

/**********************************************************************/

static void print_transfer(unsigned long in, unsigned long out)
{
  double i, o;
  char ui, uo;
  
  number_to_unit(in, &i, &ui);
  number_to_unit(out, &o, &uo);

  if (i>=1000)
    printf("  %4.4g%c", i, ui);
  else
    printf("  %4.3g%c", i, ui);

  if (o>=1000)
    printf("  %4.4g%c", o, uo);
  else
    printf("  %4.3g%c", o, uo);
}

/**********************************************************************/

static int check_constraints(const struct entry *e, int first, int last)
{
  struct tm *start;
  int s;
  time_t t;
  
  t = e->t - e->connect;
  start = localtime(&t);
  s = (start->tm_year+1900)*10000 + (start->tm_mon+1) * 100 + start->tm_mday;

  if (first && s<first) return 0;
  if (last && s>last) return 0;
 
  return 1;
}

/**********************************************************************/

static void print_entry(const struct entry *e, int trans)
{
  char s1[20], *s2;
  struct tm start, end;
  char date[11], endtime[11], starts[11];
  static char olddate[11]="";
  time_t t;

  /* nice print of connecttime */
  sec_to_string(e->connect, s1, 20);
  
  memcpy(&end, localtime(&e->t), sizeof(struct tm));
  t = e->t - e->connect;
  memcpy(&start, localtime(&t), sizeof(struct tm));
  strftime(starts,  11, "%H:%M:%S", &start);
  strftime(endtime, 11, "%H:%M:%S", &end);
  /* we need the date of the connection-start not of the connection-end */
  strftime(date, 11, "%Y-%m-%d", &start);
    
  /* a new day ? */
  if (strcmp(date, olddate)) {
    strcpy(olddate, date);
    s2 = date;
  } else {
    s2 = "";
  }

  /* print a line of the report */
  printf("%10s %8s -- %8s %12s %10s", s2, starts , endtime,  e->provider, s1);
  if (trans) 
    print_transfer(e->out, e->in);
  printf("\n");
}

/**********************************************************************/

static void print_line(int t, char c)
{
  int i;

  for (i=0; i<55; i++) putchar(c);

  if (t)
    for (i=0; i<14; i++) putchar(c);

  putchar('\n');
}

/**********************************************************************/

static void print_header(int t)
{
  printf("Date       Start       End          Provider    Connect");
  if (t) printf("   Send   Rcvd");
  putchar('\n');
}

/**********************************************************************/

static void print_provider_sums(struct totals *totals, int transfer)
{
  int i;
  char s1[20];

  /* print sums of the providers */
  print_line(transfer, '-');
  for(i=0; i<totals->provider_sum.size; i++) {
    sec_to_string(totals->provider_sum.data[i].sum, s1, 20);
    printf("Sum                             %12s %10s", 
	   totals->provider_sum.data[i].provider, s1);
    if (transfer) 
      print_transfer(totals->provider_sum.data[i].outsum, totals->provider_sum.data[i].insum);
    printf("\n");
  }
}

/**********************************************************************/

static void print_total_sum(struct totals *totals, int transfer)
{
  char s1[20];

  /* print total sum */
  print_line(transfer, '=');
  
  sec_to_string(totals->sum, s1, 20);
  printf("Sum                             %12s %10s", "total", s1);
  
  if (transfer) 
    print_transfer(totals->outsum, totals->insum);
  printf("\n");
}

/**********************************************************************/

static int process_totals(struct totals *totals, const struct entry *entry)
{
  if(!add_to_vector(&totals->provider_sum, entry)) {
    fprintf(stderr, "Can't re-allocate vector ?!\n");
    return 1;
  }
  totals->sum += entry->connect;
  totals->insum += entry->in;
  totals->outsum += entry->out;
  return 0;
}

/**********************************************************************/

static int process_file(FILE *data, struct totals *totals, int transfer, 
			int first, int last)
{
  int res;
  struct entry entry;

  while(6==(res = fscanf(data, "%ld %*s %20s %20s %ld %ld %ld\n", 
		  &entry.t, entry.provider, entry.interface, 
		  &entry.connect, &entry.out, &entry.in))) {

    if (check_constraints(&entry, first, last)) {
      print_entry(&entry, transfer);
      if (process_totals(totals, &entry)) return 1;
    }
  }

  if (res != EOF) {
    fprintf(stderr, "error scanning file\n");
    return 1;
  }
  return 0;
}

/**********************************************************************/

static const char usage[] =
"ppplog version " VERSION "\n"
"usage: ppplog [OPTION]\n"
"   or: ppplog [OPTION] time-spec\n"
"Print ppplog in human-readable form\n"
"date      := yyyy\n"
"             yyyy-mm\n"
"             yyyy-mm-dd\n"
"time-spec := date\n"
"             date..date\n"
"             ..date\n"
"             date..\n"
"Options:\n"
"  -f, --file <logfile>  use data from file ( '-' == stdin)\n"
"  -t, --transfer        print transfer-volume too (1K=1024)\n"
"  -V, --version         print version\n"
"      --help            this message\n";

/**********************************************************************/

int main(int argc, char **argv)
{
  FILE *data = 0;
  char *filename = 0;
  int transfer = FALSE;
  struct totals totals = {{0, 0, 0}, 0, 0, 0};
  const struct option longopts[] = {
    { "file",     required_argument, 0, 'f' },
    { "transfer", no_argument,       0, 't' },
    { "help",     no_argument,       0, '?' },
    { "version",  no_argument,       0, 'V' },
    { 0, 0, 0, 0 } };

  for(;;) {
    int c = getopt_long(argc, argv, "f:t?V", longopts, 0);
    if (c == -1)
      break;
    switch(c) {
    case 'f' :
      filename = optarg;
      break;
    case 't' :
      transfer = TRUE;
      break;
    case 'V' :
      puts("ppplog version " VERSION);
      return 0;
      break;
    case '?' :
      fprintf(stderr, usage);
      return 0;
      break;
    default:
	printf ("?? getopt returned character code 0%o ??\n", c);
	return 5;
    }
  }
  
  if(!init_vector(&totals.provider_sum, 5)) {
    fprintf(stderr, "Can't allocate vector ?!\n");
    return 2;
  }

  if (filename) {	/* print a file */
    if (!strcmp("-", filename)) {
      data = stdin;
    } else {
      if (!(data = fopen(filename, "r"))) {
	fprintf(stderr, "Could not open file %s: %s\n", 
		filename, strerror(errno));
	return 3;
      }
    }
    print_header(transfer);
    if (process_file(data, &totals, transfer, 0, 0)) return 1;
    fclose(data);
  } else {		/* print a region */
    int first = 0, last = 0;
    int y1, y2, m1, m2;
    int y;
    if (argv[optind]) {
      if (parse_timespec(&argv[optind], &first, &last)) {
	fprintf(stderr, usage);
	return 7;
      }
    } else {		/* process this month */
      time_t tt;
      struct tm *ts;
      
      tt = time(0);
      ts = localtime(&tt);
      
      first = (ts->tm_year+1900)*10000+(ts->tm_mon+1)*100+1;
      last =  first+30;
    }

    if (!first || !last) {
      int f, l;
      if(getfirstlast(LOGDIR "/ppplog", &f, &l)) 
	return 8;
      if (!first) first = f;
      if (!last)  last  = l;
    }

    print_header(transfer);

    y1 = first/10000;
    m1 = (first/100)%100;
    y2 = last/10000;
    m2 = (last/100)%100;

    for (y=y1; y<=y2; ++y) {
      int m, ms, me;
      ms = y==y1 ? m1 : 1;
      me = y==y2 ? m2 : 12;
      for (m=ms; m<=me; ++m) {
	char filename[FILENAME_MAX+1];
	snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%04d-%02d", y, m);
	if ((data = fopen(filename, "r"))) {
	  if (process_file(data, &totals, transfer, first, last)) return 1;
	  fclose(data);
	}
      }
    }
  }

  print_provider_sums(&totals, transfer);
  print_total_sum(&totals, transfer);

  return 0;
}
