/*
 * ppplog.c -- format log of ppp-connect-logs
 *
 * (c) 2001 M G Berberich <berberic@fmi.uni-passau.de> 
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  $Id: convert.c,v 1.1 2002/05/12 19:30:45 berberic Exp $ 
 */


/* needed for strptime prototype */
#define _GNU_SOURCE 

#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <malloc.h>
#include <getopt.h>

#ifndef TRUE
#define TRUE            1
#define FALSE           0
#endif

#ifndef VERSION
  #define VERSION "UNDEFINED"
#endif

/**********************************************************************/

struct zoneentry {
  char *name;
  int offset;
  int dst;
};

struct zoneentry zonetab[] = {
  /* Europe */
  { "UTC",       0, 0 },
  { "WET",       0, 0 },
  { "WEST",   3600, 1 },
  { "BST",    3600, 1 },
  { "IST",    3600, 1 },
  { "CET",    3600, 0 },
  { "CEST",   7200, 1 },
  { "EET",    7200, 0 },
  { "EEST",  10800, 1 },
  { "MSK",   10800, 0 },
  { "MSD",   14400, 1 },
  /* US and Canada */
  { "AST",   -14400, 0 },
  { "ADT",   -10800, 1 },
  { "EST",   -18000, 0 },
  { "EDT",   -14400, 1 },
  { "CST",   -21600, 0 },
  { "CDT",   -18000, 1 },
  { "MST",   -25200, 0 },
  { "MDT",   -21600, 1 },
  { "PST",   -28800, 0 },
  { "PDT",   -25200, 1 },
  { "HST",   -36000, 0 },
  { "AKST",  -32400, 0 },
  { "AKDT",  -28800, 1 },
  /* Australia */
  { "AEST", 36000, 0 },
  { "AEDT", 39600, 1 },
  { "ACST", 34200, 0 },
  { "ACDT", 37800, 1 },
  { "AWST", 28800, 0 },
  { 0, 0, 0}
};

static struct zoneentry *zone_to_offset(char *s)
{
  struct zoneentry *p;

  p = zonetab;
  while (p->name) {
    if (!strcmp(p->name, s)) {
      return p;
    }
    ++p;
  }
  exit(1);
}


/**********************************************************************/

static const char usage[] =
"ppplog version " VERSION "\n"
"usage: ppplogconvert [OPTION]\n"
"   or: ppplogconvert [OPTION] month-specification\n"
"convert old ppplog logfiles to new format\n"
"month-specification := yyyy-mm | yyyy mm\n"
"\n"
"  -f, --file <logfile>  use data from file\n"
"  -V, --version         print version\n"
"      --help            this message\n";

/**********************************************************************/

int main(int argc, char **argv)
{
  FILE *data = 0;
  char date[11], endtime[9], zone[5], user[9];
  char provider[21], interface[21];
  unsigned long connect, in, out;
  char s3[30];
  char *filename = 0;
  struct tm end;
  time_t t;
  int res;
  const struct option longopts[] = {
    { "file",     required_argument, 0, 'f' },
    { "help",     no_argument,       0, '?' },
    { "version",  no_argument,       0, 'V' },
    { 0, 0, 0, 0 } };

  for(;;) {
    int c = getopt_long(argc, argv, "f:t?V", longopts, 0);
    if (c == -1)
      break;
    switch(c) {
    case 'f' :
      filename = optarg;
      break;
    case 'V' :
      puts("ppplogconvert version " VERSION);
      return 0;
      break;
    case '?' :
      fprintf(stderr, usage);
      return 0;
      break;
    default:
	printf ("?? getopt returned character code 0%o ??\n", c);
	return 5;
    }
  }
  
  if (filename) {
    if (argv[optind]) {
      fprintf(stderr, "filename and month-specification"
	      "are mutual excluse!\n");
      return 4;
    }
  } else {
    if (argv[optind]) {
      filename = malloc(FILENAME_MAX+1);
      if (!filename) {
	fprintf(stderr, "can't allocate memory for filename?!\n");
	return 2;
      }
      if (argv[optind+1]) 
	snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%s-%s", 
		argv[optind], argv[optind+1]);
      else
	snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%s", argv[optind]);
    } else {
      time_t  t_t = time(0);
      struct tm *t_s = localtime (&t_t);
      char buffer[8];
      
      strftime (buffer, 8, "%Y-%m", t_s);
      
      filename = malloc(FILENAME_MAX+1);
      snprintf(filename, FILENAME_MAX, LOGDIR "/ppplog/%s", buffer);
    }
  }
  
  if (!(data = fopen(filename, "r"))) {
    fprintf(stderr, "Could not open file %s: %s\n", filename, strerror(errno));
    return 3;
  }
  
  while(9==(res = fscanf(data, "%10s %8s %4s %8s %20s %20s %ld %ld %ld\n", 
		  date, endtime, zone, user, provider, interface, 
		  &connect, &out, &in))) {

    /* some magic */
    snprintf(s3, 30, "%s %s", date, endtime);
    memset(&end, 0, sizeof(struct tm));
    strptime(s3, "%Y-%m-%d %H:%M:%S", &end); /* Should not have used this :( */
    end.tm_zone = zone; /* there seems to be a bug in zone-handling */
    end.tm_gmtoff = zone_to_offset(zone)->offset;
    end.tm_isdst = zone_to_offset(zone)->dst;
    t = mktime(&end);

    {
      char buffer[200];
      struct tm *tm;
      
      tm = localtime(&t);
      strftime(buffer, 200, "%Y-%m-%d %H:%M:%S %Z", tm);
      if (1) {
	printf("%ld %s %s %s %ld %ld %ld\n", 
	       (long)t, user, provider, interface, connect, out, in);
      } else {
	printf("%s %s %s %s %ld %ld %ld\n", buffer, user, provider, interface, connect, out, in);
      }
    }
  }

  if (res != EOF) {
    fprintf(stderr, "error scanning file\n");
    return 1;
  }

  return 0;
}
